library(rpact)
library(tidyverse)
library(gt)
library(webshot2)
library(cowplot)
library(ggplot2)
library(fmsb)
library(gsDesign)
library(mvtnorm)
library(fda.usc)
library(patchwork)


#関数simlations：SSRのシミュレーションを実行する関数
#Nmin:計画された症例数、Nmax：SSR後の最大症例数、Simnum：シミュレーション回数
#tau：情報時点（0~1）、CPcri：Promising Zoneの下限となる条件付き検出力（0~1）
#pi1：対照群のイベント発生割合、pi2：試験薬群のイベント発生割合
#stpcri：中間解析時の無効中止の基準となる条件付き検出力（0~1）
#seed：乱数のシード
#csvon：各シミュレーション結果のcsvfileでの出力の可否、（csvon=1でseed名.csvが出力される）
#erst:有効の中間解析の有無、有=1、無=0
#biason:Bias補正の解析実施の有無、有=1、無=0

simlations=function(Nmin,Nmax,simnum,tau,CPcri,pi1,pi2,stpcri,seed,erst,biason){
  myCPZSampleSizeCalculationFunction <- function(..., stage,plannedSubjects,conditionalPower,minNumberOfSubjectsPerStage,
                                                 maxNumberOfSubjectsPerStage,conditionalCriticalValue,sampleSizesPerStage,
                                                 overallRate) {
    
    #イベント数a,b、群を併合した発現割合pを計算し、割合の差の検定統計量をtestStatisticを計算
    a=round(overallRate[1]*sampleSizesPerStage[1])
    b=round(overallRate[2]*sampleSizesPerStage[2])
    p=(a+b)/(sampleSizesPerStage[1]+sampleSizesPerStage[2])
    testStatistic=(overallRate[1]-overallRate[2])/sqrt(p*(1-p)*(1/sampleSizesPerStage[1]+1/sampleSizesPerStage[2]))
    #検定統計量から条件付き検出力を算出
    cpp=round(1-pnorm((qnorm(0.975)*sqrt(minNumberOfSubjectsPerStage[2]+minNumberOfSubjectsPerStage[1])-testStatistic*sqrt(minNumberOfSubjectsPerStage[1]))/sqrt(minNumberOfSubjectsPerStage[2])-testStatistic*sqrt(minNumberOfSubjectsPerStage[2])/sqrt(minNumberOfSubjectsPerStage[1])),5)
    #関数calculateStageSubjects：条件付き検出力cpを満たすのに必要な症例数を計算
    calculateStageSubjects <- function(cp) {
      subn=(plannedSubjects[1]/testStatistic^2)*((stats::qnorm(0.975)*sqrt(plannedSubjects[2])-testStatistic*sqrt(plannedSubjects[1]))/sqrt(plannedSubjects[2]-plannedSubjects[1])+stats::qnorm(cp))^2
      return(subn)
    }
    #条件付き検出力の目標値を0.9として必要症例数を算出
    stageSubjectsCPmax <- calculateStageSubjects(cp = 0.9)
    #必要症例と最大症例数を比較して症例数を決定
    stageSubjects <- ceiling(min(max(minNumberOfSubjectsPerStage[stage],stageSubjectsCPmax), maxNumberOfSubjectsPerStage[stage]))
    if (cpp < CPcri) {
      stageSubjects <- minNumberOfSubjectsPerStage[stage]
    }
    return(stageSubjects)
  }
  n1=ceiling(tau*Nmin)
  ZF=sqrt(tau)*qnorm(0.975)+sqrt((1-tau)*tau)*qnorm(stpcri)
  if(erst==1){
  designIN <- getDesignInverseNormal(informationRates = c(tau, 1),typeOfDesign = "asOF",futilityBounds=c(ZF), bindingFutility=FALSE,kMax = 2,alpha = 0.025)}
  if(erst==0){
  designIN <- getDesignInverseNormal(typeOfDesign = "noEarlyEfficacy",kMax = 2,alpha = 0.025,beta = 0.1)}
  simCpower <- getSimulationRates(designIN,pi1 = c(pi1), pi2 = c(pi2),
                                  plannedSubjects = 2 * c(n1, Nmin),
                                  conditionalPower = 0.9,
                                  minNumberOfSubjectsPerStage = 2 * c(n1, (Nmin - n1)),
                                  maxNumberOfSubjectsPerStage = 2 * c(n1, (Nmax - n1)),
                                  calcSubjectsFunction = myCPZSampleSizeCalculationFunction,
                                  maxNumberOfIterations = simnum, showStatistics = TRUE,seed = seed)
  data <- getData.SimulationResults(simCpower)
  data2=data
  data2$a=as.numeric(data2$stagewiseRates1*data2$sampleSizesPerStage1)
  data2$b=as.numeric(data2$stagewiseRates2*data2$sampleSizesPerStage2)
  data2$b = as.integer(data2$b)
  data2$a = as.integer(data2$a)
  data2$A=as.numeric(data2$sampleSizesPerStage1)
  data2$B=as.numeric(data2$sampleSizesPerStage2)
  #ステージ累積の分母と分子を計算する
  data2=data2 %>% group_by(iterationNumber) %>% mutate(AA=cumsum(A),BB=cumsum(B),aa=cumsum(a),bb=cumsum(b))
  data2=data2 %>% group_by(iterationNumber) %>% mutate(max_stage = max(stageNumber))
  data2$row=row(data2)[,1]
  
  DIF=function(data2,biason=biason){
    options(warn=-1)
    data2_1 <- data2[data2$stageNumber ==1, ]
    ####Robertson (2023)をsupplementを参考
    n1_1 = data2_1$A   
    n0_1 = data2_1$B   
    s1_1 = data2_1$a   
    s0_1 = data2_1$b  
    ##stage 2まで到達する場合、バイアス調整を実施
    if(data2_1$max_stage==2){
      data2_2 <- data2[data2$stageNumber ==2, ]
      n1_2 = data2_2$AA    
      n0_2 = data2_2$BB    
      s1_2 = data2_2$aa    
      s0_2 = data2_2$bb     
      ptilde_1 = (s0_1 + s1_1)/(n0_1 + n1_1)
      Ihat_1 = 1/(ptilde_1*(1-ptilde_1)*(1/n0_1 + 1/n1_1))
      Z1 = (s1_1/n1_1 - s0_1/n0_1)*sqrt(Ihat_1)   # Wald test statistic
      ptilde_2 = (s0_2 + s1_2)/(n0_2 + n1_2)
      Ihat_2 = 1/(ptilde_2*(1-ptilde_2)*(1/n0_2 + 1/n1_2))
      Z2 = (s1_2/n1_2 - s0_2/n0_2)*sqrt(Ihat_2)   # Wald test statistic
     
      ### O'Brien-Fleming stopping boundaries
      original= gsDesign(k = 2, sfu = sfLDOF, test.type = 1, timing=0.5)
      ################################################################################### Calculate estimators
      # Naive (MLE)
      MLE = s1_2/n1_2 - s0_2/n0_2
      MLE_se = sqrt(1/Ihat_2)
      MLE_UCL=MLE+1.96*MLE_se
      MLE_LCL=MLE-1.96*MLE_se
      # First stage MLE
      MLE1 = s1_1/n1_1 - s0_1/n0_1
      MLE1_se = sqrt(1/Ihat_1)
      # Second stage MLE
      MLE2 = (s1_2 - s1_1)/(n1_2 - n1_1) - (s0_2 - s0_1)/(n0_2 - n0_1)
      ptilde2_star = ((s1_2-s1_1)+(s0_2-s0_1))/(n1_2+n0_2-n1_1-n0_1)
      MLE2_se = sqrt(ptilde2_star*(1-ptilde2_star)*(1/(n1_2-n1_1)+1/(n0_2-n0_1)))
      cMUE_u=NA
      cMUE_l=NA
      cMUE=NA
      UMVCUE_u=NA
      UMVCUE_l=NA
      UMVCUE=NA
      
      if(biason==1){
      #MUE (conditional)
      cov_matrix = matrix(c(1, sqrt(Ihat_1/Ihat_2), sqrt(Ihat_1/Ihat_2), 1),nrow = 2, byrow = TRUE)
      e1 = original$upper$bound[1]
      f1 = ZF
      pval_fn = function(delta) {
        mu = delta*sqrt(c(Ihat_1,Ihat_2))
        pnorm(delta*sqrt(Ihat_1) - e1) +
          pmvnorm(lower=c(f1, Z2), upper=c(e1, Inf), mean = mu, sigma = cov_matrix)[1]
        }
      f2<- function(z, theta, I1, I2, l, u, log = FALSE) {
        if (!log) {
          exp(-0.5*(z - theta*sqrt(I2))^2)*
            (pnorm(u, z*sqrt(I1/I2), sqrt((I2 - I1)/I2)) -
               pnorm(l, z*sqrt(I1/I2), sqrt((I2 - I1)/I2)))/sqrt(2*pi)
        } else {
          if (is.infinite(u)) {-0.5*(z - theta*sqrt(I2))^2 +
              pnorm((z*sqrt(I1/I2) - l)/sqrt((I2 - I1)/I2), log.p = TRUE) -log(sqrt(2*pi))
          } else {
            -0.5*(z - theta*sqrt(I2))^2 +log((pnorm(u, z*sqrt(I1/I2), sqrt((I2 - I1)/I2)) -
                    pnorm(l, z*sqrt(I1/I2), sqrt((I2 - I1)/I2)))) - log(sqrt(2*pi))}
          
        }
      }
      cmue_optim<- function(theta, k, z, I1, I2, l, u) {
        if (k == 1) {
          S1<- pnorm(l, theta*sqrt(I1), 1) + pnorm(u, theta*sqrt(I1), 1,lower.tail = FALSE)
          if (z >= u) {
            num    <- pnorm(l, theta*sqrt(I1)) + pnorm(z, theta*sqrt(I1)) - pnorm(u, theta*sqrt(I1))
          } else {
            num    <- pnorm(z, theta*sqrt(I1))
          }
          int      <- num/S1
        } else {
          if (is.infinite(u)) {
            log_S2 <- pnorm(theta*sqrt(I1) - l, log.p = TRUE)
          } else {
            log_S2 <- log(pnorm(u, theta*sqrt(I1)) - pnorm(l, theta*sqrt(I1)))
          }
          z_int    <- seq(z, qnorm(1 - 0.0001, z), length.out = 1000)
          cf2z     <- exp(f2(z_int, theta, I1, I2, l, u, log = TRUE) - log_S2)
          int      <- int.simpson2(z_int, cf2z)
          }
        (int - 0.5)^2
      }
      cMUE = optim(MLE2, cmue_optim, k = 2, z = Z2,I1 = Ihat_1, I2 = Ihat_2, l = f1, u = e1,
                   method = "Nelder-Mead")$par
      cmue_optim_l<- function(theta, k, z, I1, I2, l, u) {
        if (k == 1) {
          S1<- pnorm(l, theta*sqrt(I1), 1) + pnorm(u, theta*sqrt(I1), 1,lower.tail = FALSE)
          if (z >= u) {
            num    <- pnorm(l, theta*sqrt(I1)) + pnorm(z, theta*sqrt(I1)) - pnorm(u, theta*sqrt(I1))
          } else {
            num    <- pnorm(z, theta*sqrt(I1))
          }
          int      <- num/S1
        } else {
          if (is.infinite(u)) {
            log_S2 <- pnorm(theta*sqrt(I1) - l, log.p = TRUE)
          } else {
            log_S2 <- log(pnorm(u, theta*sqrt(I1)) - pnorm(l, theta*sqrt(I1)))
          }
          z_int    <- seq(z, qnorm(1 - 0.0001, z), length.out = 1000)
          cf2z     <- exp(f2(z_int, theta, I1, I2, l, u, log = TRUE) - log_S2)
          int      <- int.simpson2(z_int, cf2z)
        }
        (int - 0.025)^2
      }
      cMUE_l = optim(MLE2, cmue_optim_l, k = 2, z = Z2,I1 = Ihat_1, I2 = Ihat_2, l = f1, u = e1,
                   method = "Nelder-Mead")$par
      cmue_optim_u<- function(theta, k, z, I1, I2, l, u) {
        if (k == 1) {
          S1<- pnorm(l, theta*sqrt(I1), 1) + pnorm(u, theta*sqrt(I1), 1,lower.tail = FALSE)
          if (z >= u) {
            num    <- pnorm(l, theta*sqrt(I1)) + pnorm(z, theta*sqrt(I1)) - pnorm(u, theta*sqrt(I1))
          } else {
            num    <- pnorm(z, theta*sqrt(I1))
          }
          int      <- num/S1
        } else {
          if (is.infinite(u)) {
            log_S2 <- pnorm(theta*sqrt(I1) - l, log.p = TRUE)
          } else {
            log_S2 <- log(pnorm(u, theta*sqrt(I1)) - pnorm(l, theta*sqrt(I1)))
          }
          z_int    <- seq(z, qnorm(1 - 0.0001, z), length.out = 1000)
          cf2z     <- exp(f2(z_int, theta, I1, I2, l, u, log = TRUE) - log_S2)
          int      <- int.simpson2(z_int, cf2z)
        }
        (int - 0.975)^2
      }
      cMUE_u = optim(MLE2, cmue_optim_u, k = 2, z = Z2,I1 = Ihat_1, I2 = Ihat_2, l = f1, u = e1,
                     method = "Nelder-Mead")$par

      # UMVCUE (conditional)
      Ihat_2star = Ihat_2 - Ihat_1
      w1 = (Ihat_2star)^(-1)/sqrt(Ihat_1^(-1) + (Ihat_2star)^(-1))
      w2 = sqrt(Ihat_1^(-1) + (Ihat_2star)^(-1))/(Ihat_1^(-1))
      UMVCUE = MLE - w1*(dnorm(w2*(MLE - f1/sqrt(Ihat_1))) -dnorm(w2*(MLE - e1/sqrt(Ihat_1))))/
        (pnorm(w2*(MLE - f1/sqrt(Ihat_1))) -pnorm(w2*(MLE - e1/sqrt(Ihat_1))))
      }
      
    }
    ##stage 1で試験終了の場合、NAとする。
    if(data2_1$max_stage==1){
      
      MLE = s1_1/n1_1 - s0_1/n0_1
      MLE1 = s1_1/n1_1 - s0_1/n0_1
      MLE2=NA
      MLE_UCL=NA
      MLE_LCL=NA
      cMUE_u=NA
      cMUE_l=NA
      cMUE=NA
      UMVCUE_u=NA
      UMVCUE_l=NA
      UMVCUE=NA
    }
    results = data.frame(val=c(MLE,MLE_LCL,MLE_UCL,cMUE,cMUE_u,cMUE_l,UMVCUE_l,UMVCUE_u,UMVCUE),method=c('MLE',"MLE_LCL","MLE_UCL",'CMUE',"CMUE_UCL","CMUE_LCL","CUMVCUE_LCL","CUMVCUE_UCL","CUMVCUE"))
    results$sim=data2_1$iterationNumber
    results$maxst=data2_1$max_stage
    results$est=data2_1$rejectPerStage
    results$fst=data2_1$futilityPerStage
    results$val=-results$val
    return(results)
  }
  #iterationNumberごとに、Bias補正を実行し、データを整理
  temp=data2 %>% split(.$iterationNumber)  %>%  map(~DIF(data=.,biason=biason))
  tempp=data.frame(do.call("rbind", temp))
  tempp_1=tempp %>% filter(method=="MLE_LCL" |method=="CMUE_LCL" |method=="CUMVCUE_LCL") %>% mutate(lcl=val) %>% dplyr::select(sim,lcl,method) %>% mutate(method=sub("_LCL$", "", method))
  tempp_2=tempp %>% filter(method=="MLE_UCL" |method=="CMUE_UCL"|method=="CUMVCUE_UCL" ) %>% mutate(ucl=val) %>% dplyr::select(sim,ucl,method) %>% mutate(method=sub("_UCL$", "", method))
  tempp_3=tempp %>% filter(method=="MLE" |method=="CMUE" |method=="CUMVCUE") 
  tempp=merge(tempp_3,tempp_2,key=c(sim,method))
  tempp=merge(tempp,tempp_1,key=c(sim,method))
  sinodds=(pi2-pi1)
  tempp$bias=(tempp$val-sinodds)
  tempp$mse=(tempp$val-sinodds)^2
  tempp$hifuku=ifelse(tempp$ucl<sinodds & tempp$lcl>sinodds,1,0)
  outall= tempp %>% group_by(method) %>%  summarise(bias_all=mean(bias, na.rm = TRUE),mse_all=mean(mse, na.rm = TRUE))
  outest= tempp %>% filter(est==1)  %>%  group_by(method) %>%  summarise(bias_est=mean(bias, na.rm = TRUE),mse_est=mean(mse, na.rm = TRUE))
  outfst= tempp %>% filter(fst==1)  %>%  group_by(method) %>%  summarise(bias_fst=mean(bias, na.rm = TRUE),mse_fst=mean(mse, na.rm = TRUE))
  out2nd= tempp %>% filter(maxst==2) %>%   group_by(method) %>%  summarise(bias_2nd=mean(bias, na.rm = TRUE),mse_2nd=mean(mse, na.rm = TRUE),hifuku=mean(hifuku, na.rm = TRUE))
  final <- left_join(outall, outest, by = "method")
  final <- left_join(final, outfst, by = "method")
  final <- left_join(final, out2nd, by = "method")
  
 return(final)
}


######SSRの影響のみを評価するシミュレーション###########
designIN <- getDesignInverseNormal(typeOfDesign = "noEarlyEfficacy",kMax = 2,alpha = 0.025,beta = 0.1)
sampleSizeResultGS <- getSampleSizeRates(designIN, pi2 = 0.25, pi1 = 0.175)
Nmin=ceiling(sampleSizeResultGS$maxNumberOfSubjects/2)
temp0=c()
for (i in 0:100){
  print(i)
  p2=0.25-i/1000
  temp1=simlations(Nmin=Nmin,Nmax=1500,simnum=100000,tau=0.5,CPcri=0,
                   pi1=0.25,pi2=p2,stpcri=0,seed=4231,erst=0,biason=0)
  temp2=simlations(Nmin=Nmin,Nmax=1500,simnum=100000,tau=0.5,CPcri=0.1,
                 pi1=0.25,pi2=p2,stpcri=0,seed=4231,erst=0,biason=0)
  temp3=simlations(Nmin=Nmin,Nmax=1500,simnum=100000,tau=0.5,CPcri=0.2,
                   pi1=0.25,pi2=p2,stpcri=0,seed=4231,erst=0,biason=0)
  pi1=0.25
  temp1$trueodds=(pi1-p2)
  temp2$trueodds=(pi1-p2)
  temp3$trueodds=(pi1-p2)
  temp1$Promisingzone ="0%≦CP＜90%"
  temp2$Promisingzone="10%≦CP＜90%"
  temp3$Promisingzone="20%≦CP＜90%"
  temp0=rbind(temp0,temp1,temp2,temp3)
}
datain=subset(temp0, !(is.na(temp0$bias_2nd)))
p4=ggplot(datain,aes(x = trueodds, y =bias_2nd,color=Promisingzone))+geom_line(na.rm = TRUE)+xlab("リスク差の絶対値")+ylab("バイアス")+theme_light()
p4=p4+theme(plot.title = element_text(hjust = 0.5), text = element_text(size = 15),legend.position ="bottom")+
  scale_y_continuous(breaks=seq(-0.02, 0.01, 0.005), limits=c(-0.02,0.01))+geom_vline(xintercept =0.075,linetype=2)+geom_vline(xintercept =0.054,linetype=2)
ggsave("Bias_ssr.png",plot=p4, width = 12, height = 8)


#関数sim：simlationsを真のリスク差を変化させ実行する関数
#stpcri：中間解析時の無効中止の基準となる条件付き検出力（0~1）、CPcri：Promising Zoneの下限となる条件付き検出力（0~1）
#Nmax：SSR後の最大症例数、simnum：シミュレーション回数
sim=function(stpcri,CPcri,Nmax,simnum){
  tau=0.5
  #designpre <- getDesignGroupSequential(sided = 1, alpha = 0.025, beta = 0.1,informationRates = c(tau, 1), typeOfDesign = "noEarlyEfficacy")
  designpre <- getDesignInverseNormal(informationRates = c(tau, 1),typeOfDesign = "asOF",kMax = 2,alpha = 0.025)
  ZF=sqrt(tau)*designpre$criticalValues[2]+sqrt((1-tau)*tau)*qnorm(stpcri)
  designIN <- getDesignInverseNormal(informationRates = c(tau, 1),typeOfDesign = "asOF",futilityBounds=c(ZF), bindingFutility=FALSE,kMax = 2,alpha = 0.025)
  #designIN <- getDesignInverseNormal(typeOfDesign = "noEarlyEfficacy",kMax = 2,alpha = 0.025,beta = 0.1)
  sampleSizeResultGS <- getSampleSizeRates(designIN, pi2 = 0.25, pi1 = 0.175)
  Nmin=ceiling(sampleSizeResultGS$maxNumberOfSubjects/2)
  temp0=c()
  #真のオッズ比を変動させシミュレーションを実行
  for (i in 0:100){
    #print(Sys.time())
    print(i)
    p2=0.25-i/1000
    temp1=simlations(Nmin=Nmin,Nmax=Nmax,simnum=simnum,tau=0.5,CPcri=CPcri,
                     pi1=0.25,pi2=p2,stpcri=stpcri,seed=4231,erst=1,biason=1)
    pi1=0.25
    temp1$trueodds=(pi1-p2)
    temp0=rbind(temp0,temp1)
  }
  return(temp0)
}

######Bias補正のシミュレーション###########
data=sim(stpcri=0.1,CPcri=0.2,Nmax=1500,simnum=100000)
datain=data
#geom_vline(xintercept =0.075,linetype=2)+geom_vline(xintercept =0.054,linetype=2)

p41=ggplot(datain,aes(x = trueodds, y =bias_2nd,color=method))+geom_line(na.rm = TRUE)+xlab("リスク差の絶対値")+ylab("バイアス")+theme_light()+
  labs(title = "バイアス")+ theme(plot.title = element_text(hjust = 0.5), text = element_text(size = 12),legend.position = "none")+
  scale_y_continuous(breaks=seq(-0.02, 0.01, 0.005), limits=c(-0.02,0.01))+
  scale_color_manual(
    values = c("MLE" = "blue", "CMUE" = "red", "CUMVCUE" = "green"),
    breaks = c("MLE", "CUMVCUE", "CMUE")  # 凡例順を指定
  )
#ggsave("Bias.png",plot=p4, width = 8, height = 8)
options(scipen=1)
p42=ggplot(datain,aes(x = trueodds, y =mse_2nd,color=method))+geom_line(na.rm = TRUE)+xlab("リスク差の絶対値")+ylab("MSE")+theme_light()+
  labs(title = "MSE")+theme(plot.title = element_text(hjust = 0.5), text = element_text(size = 12),legend.position = "none")+
  scale_color_manual(
    values = c("MLE" = "blue", "CMUE" = "red", "CUMVCUE" = "green"),
    breaks = c("MLE", "CUMVCUE", "CMUE")  # 凡例順を指定
  )
#ggsave("MSE.png",plot=p4, width = 8, height = 8)

options(scipen=1)
p43=ggplot(datain,aes(x = trueodds, y =hifuku,color=method))+geom_line(na.rm = TRUE)+xlab("リスク差の絶対値")+ylab("被覆確率")+theme_light()+
  labs(title = "被覆確率")+theme(plot.title = element_text(hjust = 0.5), text = element_text(size = 12),legend.position = "bottom")+
  scale_y_continuous(breaks=seq(0.9, 1, 0.01), limits=c(0.9,1))+
  scale_color_manual(
    values = c("MLE" = "blue", "CMUE" = "red", "CUMVCUE" = "green"),
    breaks = c("MLE", "CUMVCUE", "CMUE")  # 凡例順を指定
    )+ labs(color = "補正方法")
#ggsave("hifuku.png",plot=p4, width = 8, height = 8)
p4=p41/p42/p43
ggsave("Bias_adjuted.png",plot=p4, width = 8, height = 12)







