library(rpact)
library(tidyverse)
library(gt)
library(webshot2)
library(cowplot)
library(ggplot2)

getwd()
#関数simlations：SSRのシミュレーションを実行する関数
#Nmin:計画された症例数、Nmax：SSR後の最大症例数、Simnum：シミュレーション回数
#tau：情報時点（0~1）、CPcri：Promising Zoneの下限となる条件付き検出力（0~1）
#pi1：対照群のイベント発生割合、pi2：試験薬群のイベント発生割合
#stpcri：中間解析時の無効中止の基準となる条件付き検出力（0~1）
#seed：乱数のシード
#csvon：各シミュレーション結果のcsvfileでの出力の可否、（csvon=1でseed名.csvが出力される）

simlations=function(Nmin,Nmax,simnum,tau,CPcri,pi1,pi2,stpcri,seed,csvon){
  #関数myCPZSampleSizeCalculationFunction ：Rpactの関数getSimulationRatesでSSRの方法を規定する関数
  #各変数(stage、plannedSubjects等)は getSimulationRatesで出力される変数を持ち越している
  myCPZSampleSizeCalculationFunction <- function(..., stage,plannedSubjects,conditionalPower,minNumberOfSubjectsPerStage,
                                                 maxNumberOfSubjectsPerStage,conditionalCriticalValue,sampleSizesPerStage,
                                                 overallRate) {
    #イベント数a,b、群を併合した発現割合pを計算し、割合の差の検定統計量をtestStatisticを計算  
    a=overallRate[1]*sampleSizesPerStage[1]
    b=overallRate[2]*sampleSizesPerStage[2]
    p=(a+b)/(sampleSizesPerStage[1]+sampleSizesPerStage[2])
    testStatistic=(overallRate[1]-overallRate[2])/sqrt(p*(1-p)*(1/sampleSizesPerStage[1]+1/sampleSizesPerStage[2]))
    #検定統計量から条件付き検出力を算出
    cpp=round(1-pnorm((qnorm(0.975)*sqrt(minNumberOfSubjectsPerStage[2]+minNumberOfSubjectsPerStage[1])-testStatistic*sqrt(minNumberOfSubjectsPerStage[1]))/sqrt(minNumberOfSubjectsPerStage[2])-testStatistic*sqrt(minNumberOfSubjectsPerStage[2])/sqrt(minNumberOfSubjectsPerStage[1])),5)
    #関数calculateStageSubjects：条件付き検出力cpを満たすのに必要な症例数を計算
    calculateStageSubjects <- function(cp) {
      subn=(plannedSubjects[1]/testStatistic^2)*((stats::qnorm(0.975)*sqrt(plannedSubjects[2])-testStatistic*sqrt(plannedSubjects[1]))/sqrt(plannedSubjects[2]-plannedSubjects[1])+stats::qnorm(cp))^2 
      return(subn)
    }
    #条件付き検出力の目標値を0.9として必要症例数を算出
    stageSubjectsCPmax <- calculateStageSubjects(cp = 0.9)
    #必要症例と最大症例数を比較して症例数を決定
    stageSubjects <- ceiling(min(max(
      minNumberOfSubjectsPerStage[stage],
      stageSubjectsCPmax
    ), maxNumberOfSubjectsPerStage[stage]))
    #条件付き検出力がPromising Zoneの範囲にある場合のみSSRを実行
    if (cpp < CPcri) {
      stageSubjects <- minNumberOfSubjectsPerStage[stage]
    }
    return(stageSubjects)
  }
  #n1：第一ステージの症例数
  n1=ceiling(tau*Nmin)
  #中間解析時の無効中止の基準を検定統計量スケールに変換
  ZF=sqrt(tau)*qnorm(0.975)+sqrt((1-tau)*tau)*qnorm(stpcri)
  #getDesignInverseNormal:逆正規法で群逐次デザインを実行する関数
  #https://rdrr.io/cran/rpact/man/getDesignInverseNormal.html
  designIN <- getDesignInverseNormal(informationRates = c(tau, 1),
                                     typeOfDesign = "asOF",futilityBounds=c(ZF), bindingFutility=FALSE,
                                     kMax = 2,alpha = 0.025)
  #getSimulationRates：割合の差に関する、シミュレーションを実行する関数
  #https://rdrr.io/cran/rpact/man/getSimulationRates.html
  simCpower <- getSimulationRates(designIN,pi1 = c(pi1), pi2 = c(pi2),
                                  plannedSubjects = 2 * c(n1, Nmin),
                                  conditionalPower = 0.9,
                                  minNumberOfSubjectsPerStage = 2 * c(n1, (Nmin - n1)),
                                  maxNumberOfSubjectsPerStage = 2 * c(n1, (Nmax - n1)),
                                  calcSubjectsFunction = myCPZSampleSizeCalculationFunction,
                                  maxNumberOfIterations = simnum, showStatistics = TRUE,seed = seed)
  #シミュレーション結果を代入
  data <- getData(simCpower)
  #
  data$cp=round(1-pnorm((designIN$criticalValues[2]*sqrt(Nmin)-data$testStatisticsPerStage*sqrt(n1))/sqrt(Nmin-n1)-data$testStatisticsPerStage*sqrt(Nmin-n1)/sqrt(n1)),5)
  data$cpm=round(1-pnorm((designIN$criticalValues[2]*sqrt(Nmin)-data$testStatisticsPerStage*sqrt(n1))/sqrt(Nmin-n1)-data$testStatisticsPerStage*sqrt(Nmax-n1)/sqrt(n1)),5)
  #第1ステージでの無効中止の割合
  Fst1=format(round(sum(data$futilityPerStage)/simnum,3),nsmall = 3)
  #第2ステージでの有意にならなかった割合
  Fst2=format(round(sum(1-data[data$stageNumber==2,]$rejectPerStage)/simnum,3),nsmall = 3)
  #第1ステージでの有効中止の割合
  Ast=format(round(sum(data[data$stageNumber==1,]$rejectPerStage)/simnum,3),nsmall = 3)
  #第2ステージでの有意になった割合
  st=format(round(sum(data[data$stageNumber==2,]$rejectPerStage)/simnum,3),nsmall = 3)
  #試験全体での有意になった割合
  Tst=format(round(sum(data$rejectPerStage)/simnum,3),nsmall = 3)
  ncol=""
  #試験全体での有意にならなかった割合
  TFst=format(round((sum(data$futilityPerStage)+sum(1-data[data$stageNumber==2,]$rejectPerStage))/simnum,3),nsmall = 3)
  STT=rbind(Tst,Ast,st,TFst,Fst1,Fst2)
  
  #第2ステージまで到達した場合、第1ステージのレコードを削除し、1シミュレーション1レコードとする
  data1=(data[data$stageNumber==2 | data$futilityPerStage==1 | (data$stageNumber==1 & data$rejectPerStage==1),]) 
  #シミュレーション結果をcsvとして出力
  if(csvon==1){write.csv(data1, paste(seed,".csv",sep=""))} 
  
  #症例数の要約統計量を算出
  mean=as.integer(mean(data1$numberOfCumulatedSubjects)/2)
  median=as.integer(median(data1$numberOfCumulatedSubjects)/2)
  min=min(data1$numberOfCumulatedSubjects)/2
  max=max(data1$numberOfCumulatedSubjects)/2
  NNN=rbind(paste0(mean,"(",median,")"),paste0(min,"-",max))
  
  #計画された症例数で第2ステージ完了の到達割合
  Nonnoninc=round(length(data1[data1$numberOfCumulatedSubjects==n1*2,]$numberOfCumulatedSubjects)/simnum,3)
  #当初の症例数に対して症例数が増加した割合
  Noninc=round(length(data1[data1$numberOfCumulatedSubjects==Nmin*2,]$numberOfCumulatedSubjects)/simnum,3)
  #最大症例数の到達割合
  maxinc=round(length(data1[data1$numberOfCumulatedSubjects==Nmax*2,]$numberOfCumulatedSubjects)/simnum,3)
  inc=format(round(length(data1[data1$numberOfCumulatedSubjects>Nmin*2,]$numberOfCumulatedSubjects)/simnum,3),nsmall = 3)
  incc=format(rbind(Nonnoninc,Noninc,maxinc),nsmall = 3)
  #真のオッズを計算
  sinodds=log(1/((pi1*(1-pi2))/((1-pi1)*pi2)))
  #対数オッズ比のバイアス、MSE、被覆確率の計算  
  data1$odds=(data1$overallRate2*(1-data1$overallRate1))/(data1$overallRate1*(1-data1$overallRate2))
  data1$se=sqrt(1/(data1$overallRate2*data1$numberOfCumulatedSubjects/2)+1/((1-data1$overallRate1)*data1$numberOfCumulatedSubjects/2)+1/(data1$overallRate1*data1$numberOfCumulatedSubjects/2)+1/((1-data1$overallRate2)*data1$numberOfCumulatedSubjects/2))
  data1$lodds=log(data1$odds)
  data1$ucl=log(data1$odds)+qnorm(0.975)*data1$se
  data1$lcl=log(data1$odds)-qnorm(0.975)*data1$se
  data1$hifuku=ifelse(data1$lcl<sinodds & sinodds<data1$ucl,1,0)
  data1$bias=(data1$lodds-sinodds)
  data1$mse=(data1$lodds-sinodds)^2
  
  #第1ステージで無効中止の場合のバイアス、MSE、被覆確率
  temp=(data1[data1$futilityPerStage==1,]) 
  odds1=format(rbind(round(mean(temp$lodds),3),round(mean(temp$bias),3),round(mean(temp$mse),3),round(mean(temp$hifuku),3)),nsmall = 3)
  #第1ステージで有効中止の場合のバイアス、MSE、被覆確率
  temp=(data1[data1$stageNumber==1 & data1$rejectPerStage==1,]) 
  odds2=format(rbind(round(mean(temp$lodds),3),round(mean(temp$bias),3),round(mean(temp$mse),3),round(mean(temp$hifuku),3)),nsmall = 3)
  #第2ステージまで到達した場合のバイアス、MSE、被覆確率
  temp=(data1[data1$stageNumber==2,]) 
  odds3=format(rbind(round(mean(temp$lodds),3),round(mean(temp$bias),3),round(mean(temp$mse),3),round(mean(temp$hifuku),3)),nsmall = 3)
  #試験全体のバイアス、MSE、被覆確率
  temp=data1
  odds4=format(rbind(round(mean(temp$lodds),3),round(mean(temp$bias),3),round(mean(temp$mse),3),round(mean(temp$hifuku),3)),nsmall = 3)
  sinodds=format(round(sinodds,3),nsmall = 3)
  final=rbind(STT,NNN,incc,inc,sinodds,ncol,odds1,ncol,odds2,ncol,odds3,ncol,odds4)
  return(final)
}

#関数Main_simlation：simlationsを条件を（対立仮説下、対立仮説下（最低差）、帰無仮説下）で実行する関数
#pi1：対照群のイベント発生割合、pi2：試験薬群のイベント発生割合、、pi3：試験薬群のイベント発生割合（最低差）
#simnum：シミュレーション回数、tau：情報時点（0~1）、Nmax：SSR後の最大症例数、Nmin:計画された症例数
#stpcri：中間解析時の無効中止の基準となる条件付き検出力（0~1）、CPcri：Promising Zoneの下限となる条件付き検出力（0~1）
#tablename:出力ファイル名

Main_simlation=function(pi1,pi2,pi3,simnum,tau,Nmax,stpcri,CPcri,tablename){
  #getDesignGroupSequential：SSRなし群逐次デザインを実行する関数
  #https://rdrr.io/cran/rpact/man/getDesignGroupSequential.html
  designpre <- getDesignGroupSequential(sided = 1, alpha = 0.025, beta = 0.1,informationRates = c(tau, 1), typeOfDesign = "asOF")
  #中間解析時の無効中止の基準を検定統計量スケールに変換
  ZF=sqrt(tau)*designpre$criticalValues[2]+sqrt((1-tau)*tau)*qnorm(stpcri)
  #計画された症例数の計算
  design <- getDesignGroupSequential(sided = 1, alpha = 0.025, beta = 0.1,informationRates = c(tau, 1), typeOfDesign = "asOF",futilityBounds=c(ZF), bindingFutility=FALSE)
  sampleSizeResultGS <- getSampleSizeRates(design, pi2 = 0.25, pi1 = 0.175)
  Nmin=ceiling(sampleSizeResultGS$maxNumberOfSubjects/2)
  
  #対立仮説下SSRあり
  temp1=simlations(Nmin=Nmin,Nmax=Nmax,simnum=simnum,tau=tau,CPcri=CPcri,
                   pi1=pi1,pi2=pi2,stpcri=stpcri,seed=1192,csvon=1)
  #対立仮説下SSRなし
  temp2=simlations(Nmin=Nmin,Nmax=Nmax,simnum=simnum,tau=tau,CPcri=1.3,
                   pi1=pi1,pi2=pi2,stpcri=stpcri,seed=794,csvon=1)
  #対立仮説下（最低差）SSRあり
  temp3=simlations(Nmin=Nmin,Nmax=Nmax,simnum=simnum,tau=tau,CPcri=CPcri,
                   pi1=pi1,pi2=pi3,stpcri=stpcri,seed=710,csvon=1)
  #対立仮説下（最低差）SSRなし
  temp4=simlations(Nmin=Nmin,Nmax=Nmax,simnum=simnum,tau=tau,CPcri=1.3,
                   pi1=pi1,pi2=pi3,stpcri=stpcri,seed=1945,csvon=1)
  #帰無仮説下SSRあり
  temp5=simlations(Nmin=Nmin,Nmax=Nmax,simnum=simnum,tau=tau,CPcri=CPcri,
                   pi1=pi1,pi2=pi1,stpcri=stpcri,seed=1603,csvon=1)
  #帰無仮説下SSRなし
  temp6=simlations(Nmin=Nmin,Nmax=Nmax,simnum=simnum,tau=tau,CPcri=1.3,
                   pi1=pi1,pi2=pi1,stpcri=stpcri,seed=1914,csvon=1)
  
  #対立仮説下SSRなし（N=1020）
  temp7=simlations(Nmin=1020,Nmax=2000,simnum=simnum,tau=0.5,CPcri=1.3,
                   pi1=pi1,pi2=pi2,stpcri=stpcri,seed=1939,csvon=1)
  #対立仮説下（最低差）SSRなし（N=1020）
  temp8=simlations(Nmin=1020,Nmax=2000,simnum=simnum,tau=0.5,CPcri=1.3,
                   pi1=pi1,pi2=pi3,stpcri=stpcri,seed=1868,csvon=1)
  #帰無仮説下SSRなし（N=1020）
  temp9=simlations(Nmin=1020,Nmax=2000,simnum=simnum,tau=0.5,CPcri=1.3,
                   pi1=pi1,pi2=pi1,stpcri=stpcri,seed=1338,csvon=1)
  
  final=cbind(temp1,temp2,temp7,temp3,temp4,temp8,temp5,temp6,temp9)
  
  #Tableの作成
  c1=c(rep("1. 検出力及び第1種の過誤率",3),rep("2. 無効中止 ",3),
       rep("3. 症例数（全シミュレーションの平均症例数、中央値、範囲）",2),
       rep("4. 治験が可能な最小又は最大の症例数を達成した回数の割合",3),
       rep("5. 当初の症例数に対して症例数が増加した回数の割合及び/又は特定の水準を超えて症例数が増加した回数の割合（SSR）",1),
       rep("6. 中止決定時及び最終解析時の推定治療効果（対数オッズ比スケール）",21)
  )
  c2=c("有効中止（全体：検出力/第1種の過誤確率）","　第1ステージ","　第2ステージ",
       "無効中止","　第1ステージ","　第2ステージ","平均症例数 (中央値)","最小-最大",
       "第1ステージで有効又は無効で中止した割合","計画された症例数で第2ステージ完了の到達割合","最大例数の到達割合",
       "当初の症例数に対して症例数が増加した割合","真値","無効中止時（第1ステージ）",
       "　平均","  バイアス","  MSE","  被覆確率","有効中止時（第1ステージ）","　平均","　バイアス","　MSE","　被覆確率",
       "第1+第2ステージ（第2ステージ完了時）","　平均","　バイアス","　MSE","　被覆確率","Unconditional","　平均","　バイアス","　MSE","　被覆確率")
  c3=paste("OR=0.636(症例数設計の仮定)の仮定で検出力が90%になるように症例数を設定：1群N=",Nmin,sep="")
  c4=paste("OR=0.731(症例数設計の仮定より下振れ)の仮定で検出力が90%になるように症例数を設定：1群N=","1020",sep="")
  c5=paste()
  c6=tablename
  final1=data.frame(final)
  final1$g=c1
  final1$r=c2
  tableput=final1 %>% gt(groupname_col="g",rowname_col="r") %>%
    cols_label(X1=md("SSRあり"),X2=md("SSRなし"),X3=md("SSRなし"),X4=md("SSRあり"),
               X5=md("SSRなし"),X6=md("SSRなし"),X7=md("SSRあり"),X8=md("SSRなし"),X9=md("SSRなし")) %>%
    tab_spanner(label = "OR=0.636(期待する効果)", columns = c("X1","X2","X3")) %>%
    tab_spanner(label = "OR=0.731(期待する最低差)", columns = c("X4", "X5","X6")) %>%
    tab_spanner(label = "OR=1(帰無仮説)", columns = c("X7", "X8","X9")) %>%
    tab_footnote(footnote =c3,
                 locations = cells_column_labels(columns = c(X1,X2,X4,X5,X7,X8))) %>%
    tab_footnote(footnote =c4,
                 locations = cells_column_labels(columns = c(X3,X6,X9))) %>% 
    tab_options(table.border.bottom.width = 0,table.font.size=px(8)) %>%
    tab_source_note(source_note = md(c5)) %>%
    tab_header(title =tablename)
  gtsave(tableput,paste(tablename,".docx",sep=""))
  gtsave(tableput,paste(tablename,".html",sep=""))
  
  #Histogram
  d1=read.csv("1192.csv")
  d2=read.csv("794.csv")
  d3=read.csv("1939.csv")
  d4=read.csv("710.csv")
  d5=read.csv("1945.csv")
  d6=read.csv("1868.csv")
  d7=read.csv("1603.csv")
  d8=read.csv("1914.csv")
  d9=read.csv("1338.csv")
  
  d1$Design="SSRあり（N=685）"
  d2$Design="SSRなし（N=685）"
  d3$Design="SSRなし（N=1020）"
  d4$Design="SSRあり（N=685）"
  d5$Design="SSRなし（N=685）"
  d6$Design="SSRなし（N=1020）"
  d7$Design="SSRあり（N=685）"
  d8$Design="SSRなし（N=685）"
  d9$Design="SSRなし（N=1020）"
  dd1=rbind(d1,d2,d3)
  dd2=rbind(d4,d5,d6)
  dd3=rbind(d7,d8,d9)
  
  dd1$Setting="真のオッズ比=0.636（期待する効果）"
  dd2$Setting="真のオッズ比=0.731（期待する最低差）"
  dd3$Setting="真のオッズ比=1（帰無仮説下）"

  
  dd=rbind(dd1,dd2,dd3)
  dd1$Design<- factor(dd1$Design, levels = c("SSRあり（N=685）","SSRなし（N=685）", "SSRなし（N=1020）"))
  dd2$Design<- factor(dd2$Design, levels = c("SSRあり（N=685）","SSRなし（N=685）", "SSRなし（N=1020）"))
  dd3$Design<- factor(dd3$Design, levels = c("SSRあり（N=685）","SSRなし（N=685）", "SSRなし（N=1020）"))
  dd$Design<- factor(dd$Design, levels = c("SSRあり（N=685）","SSRなし（N=685）", "SSRなし（N=1020）"))
  dd$Setting<- factor(dd$Setting, levels = c("真のオッズ比=1（帰無仮説下）","真のオッズ比=0.636（期待する効果）","真のオッズ比=0.731（期待する最低差）"))
  g=ggplot(dd, aes(x =numberOfCumulatedSubjects/2,y = after_stat(count / sum(count)),fill = Design, color = Design))+
    geom_histogram(position = "identity", alpha = .3,binwidth =30)+facet_wrap(~Setting)+theme_light()+xlab("症例数")+ylab("割合")+scale_x_continuous(breaks=seq(300,1500,300))+
    theme(legend.position="bottom",legend.title=element_blank(),strip.text=element_text(colour="black"),text = element_text(size = 18))
  ggsave(filename=paste(tablename,"Histogram.png"),plot=g,dpi=300,width = 16, height = 9)
}

#関数Main_simlation：simlationsを真のオッズ比を変化させ実行する関数
#stpcri：中間解析時の無効中止の基準となる条件付き検出力（0~1）、CPcri：Promising Zoneの下限となる条件付き検出力（0~1）
#Nmax：SSR後の最大症例数、simnum：シミュレーション回数
sim=function(stpcri,CPcri,Nmax,simnum){
  tau=0.5
  designpre <- getDesignGroupSequential(sided = 1, alpha = 0.025, beta = 0.1,informationRates = c(tau, 1), typeOfDesign = "asOF")
  ZF=sqrt(tau)*designpre$criticalValues[2]+sqrt((1-tau)*tau)*qnorm(stpcri)
  design <- getDesignGroupSequential(
    sided = 1, alpha = 0.025, beta = 0.1,
    informationRates = c(tau, 1), typeOfDesign = "asOF",
    futilityBounds=c(ZF), bindingFutility=FALSE)
  sampleSizeResultGS <- getSampleSizeRates(design, pi2 = 0.25, pi1 = 0.175)
  Nmin=ceiling(sampleSizeResultGS$maxNumberOfSubjects/2)
  temp0=c()
  #真のオッズ比を変動させシミュレーションを実行
  for (i in 0:107){
    p2=0.25-i/1000
    temp1=simlations(Nmin=Nmin,Nmax=Nmax,simnum=simnum,tau=0.5,CPcri=CPcri,
                     pi1=0.25,pi2=p2,stpcri=stpcri,seed=4231,csvon=0)
    temp2=simlations(Nmin=1020,Nmax=Nmax,simnum=simnum,tau=0.5,CPcri=1.3,
                     pi1=0.25,pi2=p2,stpcri=stpcri,seed=352,csvon=0)
    temp3=simlations(Nmin=Nmin,Nmax=Nmax,simnum=simnum,tau=0.5,CPcri=1.3,
                     pi1=0.25,pi2=p2,stpcri=stpcri,seed=433,csvon=0)
    pi1=0.25
    trueodds=(1/((pi1*(1-p2))/((1-pi1)*p2)))
    temp=data.frame(as.data.frame(t(temp1)),as.data.frame(t(temp2)),as.data.frame(t(temp3)))
    temp$trueodds=trueodds
    temp0=rbind(temp0,temp)
  }
  return(temp0)
}

#各種グラフを作成する関数
makeg=function(data,figname){
  #平均症例数と検出力  
  dd1=data %>% 
    pivot_longer(cols = !trueodds, # 対象列
                 names_to = "cat", 
                 values_to = "VAL") 
  dd2=subset(dd1,dd1$cat=="V7" | dd1$cat=="V7.1" | dd1$cat=="V7.2")
  dd2$Design<-ifelse(dd2$cat=="V7","SSRあり（N=685）",
                     ifelse(dd2$cat=="V7.1","SSRなし（N=1020）","SSRなし（N=685）"))
  dd2$Design<- factor(dd2$Design, levels = c("SSRあり（N=685）","SSRなし（N=685）", "SSRなし（N=1020）"))
  dd2$samplesize=as.numeric(gsub("\\(.*\\)", "", dd2$VAL))
  dd3=subset(dd1,dd1$cat=="Tst" | dd1$cat=="Tst.1" | dd1$cat=="Tst.2")
  dd3$Design<-ifelse(dd2$cat=="Tst","SSRあり（N=685）",
                     ifelse(dd2$cat=="Tst.1","SSRなし（N=1020）","SSRなし（N=685）"))
  dd3$Design<- factor(dd2$Design, levels = c("SSRあり（N=685）","SSRなし（N=685）", "SSRなし（N=1020）"))
  dd3$Power=as.numeric(dd3$VAL)
  dd4=merge(dd2,dd3,by=c("trueodds","Design"))
  dd4$Design1=paste("平均症例数：",dd4$Design)
  dd4$Design2=paste("検出力：",dd4$Design)
  
  p1 = ggplot(dd4)+ geom_line(aes(x = trueodds, y = samplesize/1000,color =Design1))+
    geom_area(aes(x = trueodds, y =Power,fill=Design2),alpha=0.2,position = 'identity')+
    scale_y_continuous(limits = c(0,1),breaks=seq(0,1,0.1),name = "検出力", sec.axis =sec_axis(~ . *1000, name = "平均症例数",breaks=seq(0,1000,100)))+xlab("オッズ比")+
    scale_x_continuous(limits = c(0.5,1),breaks=seq(0.5,1,0.1))+
    theme_light()+xlab("オッズ比")+geom_vline(xintercept =0.636,linetype=2)+geom_vline(xintercept =0.731,linetype=2)+theme(legend.position="bottom",legend.title=element_blank(),text = element_text(size = 14))
  ggsave(filename=paste0(figname,"_1.png"),plot=p1,dpi=300,width = 16, height = 9)
  
  #増加割合
  dd2=subset(dd1,dd1$cat=="inc" |dd1$cat=="maxinc")
  dd2$Design<-ifelse(dd2$cat=="inc","当初の症例数に対して症例数が増加した割合","最大例数の到達割合")
  dd2$Propotion=as.numeric(dd2$VAL)
  p2=ggplot(dd2, aes(x = trueodds, y = Propotion,color =Design))+geom_line()+
    xlab("オッズ比")+ylab("割合")+geom_vline(xintercept =0.636,linetype=2)+theme_light()+
    geom_vline(xintercept =0.731,linetype=2)+theme(legend.position="bottom",legend.title=element_blank(),text = element_text(size = 18))
  ggsave(filename=paste0(figname,"_2.png"),plot=p2,dpi=300,width = 16, height = 9)
  
  #バイアス
  dd2=subset(dd1,dd1$cat=="V16" | dd1$cat=="V21" | dd1$cat=="V26"| dd1$cat=="V31" |
               dd1$cat=="V16.1" | dd1$cat=="V21.1" | dd1$cat=="V26.1"| dd1$cat=="V31.1"|
               dd1$cat=="V16.2" | dd1$cat=="V21.2" | dd1$cat=="V26.2"| dd1$cat=="V31.2")
  dd2$Design<-ifelse(dd2$cat=="V16" |dd2$cat=="V21" |dd2$cat=="V26"| dd2$cat=="V31" ,"SSRあり（N=685）",
                     ifelse(dd2$cat=="V16.1" |dd2$cat=="V21.1" |dd2$cat=="V26.1"| dd2$cat=="V31.1" ,"SSRなし（N=1020）","SSRなし（N=685）"))
  dd2$Setting<-ifelse(dd2$cat=="V16" |dd2$cat=="V16.1" |dd2$cat=="V16.2","無効中止時（第1ステージ）",
                      ifelse(dd2$cat=="V21" |dd2$cat=="V21.1" |dd2$cat=="V21.2","有効中止時（第1ステージ）",
                             ifelse(dd2$cat=="V26" |dd2$cat=="V26.1" |dd2$cat=="V26.2","第1+第2ステージ（第2ステージ完了時）","Unconditional")))
  dd2$Design<- factor(dd2$Design, levels = c("SSRあり（N=685）","SSRなし（N=685）", "SSRなし（N=1020）"))
  dd2$Setting<- factor(dd2$Setting, levels = c("無効中止時（第1ステージ）","有効中止時（第1ステージ）", "第1+第2ステージ（第2ステージ完了時）","Unconditional"))
  dd2$Bias=as.numeric(dd2$VAL)
  p4=ggplot(dd2,aes(x = trueodds, y = Bias,color =Design))+theme_light()+geom_line()+facet_wrap(~ Setting)+
    xlab("オッズ比")+geom_vline(xintercept =0.636,linetype=2)+geom_vline(xintercept =0.731,linetype=2)+
    ylab("バイアス") +theme(legend.position="bottom",legend.title=element_blank(),strip.text=element_text(colour="black"),text = element_text(size = 18))
  ggsave(filename=paste0(figname,"_3.png"),plot=p4,dpi=300,width = 16, height = 9)
  
  ##MSE
  dd2=subset(dd1,dd1$cat=="V17" | dd1$cat=="V22" | dd1$cat=="V27"| dd1$cat=="V32" |
               dd1$cat=="V17.1" | dd1$cat=="V22.1" | dd1$cat=="V27.1"| dd1$cat=="V32.1"|
               dd1$cat=="V17.2" | dd1$cat=="V22.2" | dd1$cat=="V27.2"| dd1$cat=="V32.2")
  dd2$Design<-ifelse(dd2$cat=="V17" |dd2$cat=="V22" |dd2$cat=="V27"| dd2$cat=="V32" ,"SSRあり（N=685）",
                     ifelse(dd2$cat=="V17.1" |dd2$cat=="V22.1" |dd2$cat=="V27.1"| dd2$cat=="V32.1" ,"SSRなし（N=1020）","SSRなし（N=685）"))
  dd2$Setting<-ifelse(dd2$cat=="V17" |dd2$cat=="V17.1" |dd2$cat=="V17.2","無効中止時（第1ステージ）",
                      ifelse(dd2$cat=="V22" |dd2$cat=="V22.1" |dd2$cat=="V22.2","有効中止時（第1ステージ）",
                             ifelse(dd2$cat=="V27" |dd2$cat=="V27.1" |dd2$cat=="V27.2","第1+第2ステージ（第2ステージ完了時）","Unconditional")))
  dd2$Design<- factor(dd2$Design, levels = c("SSRあり（N=685）","SSRなし（N=685）", "SSRなし（N=1020）"))
  dd2$Setting<- factor(dd2$Setting, levels = c("無効中止時（第1ステージ）","有効中止時（第1ステージ）", "第1+第2ステージ（第2ステージ完了時）","Unconditional"))
  dd2$Bias=as.numeric(dd2$VAL)
  p4=ggplot(dd2,aes(x = trueodds, y = Bias,color =Design))+theme_light()+geom_line()+facet_wrap(~ Setting)+
    xlab("オッズ比")+geom_vline(xintercept =0.636,linetype=2)+geom_vline(xintercept =0.731,linetype=2)+
    ylab("MSE") +theme(legend.position="bottom",legend.title=element_blank(),strip.text=element_text(colour="black"),text = element_text(size = 18))
  ggsave(filename=paste0(figname,"_4.png"),plot=p4,dpi=300,width = 16, height = 9)
  
  ##被覆確率
  dd2=subset(dd1,dd1$cat=="V18" | dd1$cat=="V23" | dd1$cat=="V28"| dd1$cat=="V33" |
               dd1$cat=="V18.1" | dd1$cat=="V23.1" | dd1$cat=="V28.1"| dd1$cat=="V33.1"|
               dd1$cat=="V18.2" | dd1$cat=="V23.2" | dd1$cat=="V28.2"| dd1$cat=="V33.2")
  dd2$Design<-ifelse(dd2$cat=="V18" |dd2$cat=="V23" |dd2$cat=="V28"| dd2$cat=="V33" ,"SSRあり（N=685）",
                     ifelse(dd2$cat=="V18.1" |dd2$cat=="V23.1" |dd2$cat=="V28.1"| dd2$cat=="V33.1" ,"SSRなし（N=1020）","SSRなし（N=685）"))
  dd2$Setting<-ifelse(dd2$cat=="V18" |dd2$cat=="V18.1" |dd2$cat=="V18.2","無効中止時（第1ステージ）",
                      ifelse(dd2$cat=="V23" |dd2$cat=="V23.1" |dd2$cat=="V23.2","有効中止時（第1ステージ）",
                             ifelse(dd2$cat=="V28" |dd2$cat=="V28.1" |dd2$cat=="V28.2","第1+第2ステージ（第2ステージ完了時）","Unconditional")))
  dd2$Design<- factor(dd2$Design, levels = c("SSRあり（N=685）","SSRなし（N=685）", "SSRなし（N=1020）"))
  dd2$Setting<- factor(dd2$Setting, levels = c("無効中止時（第1ステージ）","有効中止時（第1ステージ）", "第1+第2ステージ（第2ステージ完了時）","Unconditional"))
  dd2$Bias=as.numeric(dd2$VAL)
  p5=ggplot(dd2,aes(x = trueodds, y = Bias,color =Design))+theme_light()+geom_line()+facet_wrap(~ Setting)+
    xlab("オッズ比")+geom_vline(xintercept =0.636,linetype=2)+geom_vline(xintercept =0.731,linetype=2)+
    ylab("被覆確率") +theme(legend.position="bottom",legend.title=element_blank(),strip.text=element_text(colour="black"),text = element_text(size = 18))
  ggsave(filename=paste0(figname,"_5.png"),plot=p5,dpi=300,width = 16, height = 9)
  return()}

#シミュレーションの実行
Main_simlation(pi1=0.25,pi2=0.175,pi3=0.196,simnum=100000,tau=0.5,Nmax=1500,stpcri=0.1,CPcri=0.2,tablename="000_pi_1=0.25,Nmax=1500,tau=0.5,promU=0.2")
Main_simlation(pi1=0.25,pi2=0.175,pi3=0.196,simnum=1000,tau=0.5,Nmax=1500,stpcri=0.1,CPcri=0.1,tablename="000_pi_1=0.25,Nmax=1500,tau=0.5,promU=0.1")
Main_simlation(pi1=0.25,pi2=0.175,pi3=0.196,simnum=1000,tau=0.5,Nmax=1500,stpcri=0.2,CPcri=0.2,tablename="000_pi_1=0.25,Nmax=1500,tau=0.5,promU=0.2,SP=0.2")

makeg(data=sim(stpcri=0.1,CPcri=0.2,Nmax=1500,simnum=1000),figname="fig1")
makeg(data=sim(stpcri=0.1,CPcri=0.1,Nmax=1500,simnum=10000),figname="fig2")
makeg(data=sim(stpcri=0.2,CPcri=0.2,Nmax=1500,simnum=100),figname="fig3")
